### Load libraries
library(ggplot2)

### Import data for bursts & clean up data
data <- read.csv("../Documents/R codes/MEA/20240222_264-A_cJUN_Baseline_after_MC(006)(000)_network_burst_list.csv")
burst_data <- data.frame(
  BurstStartTime = data$'Time..s.', #time at which burst starts
  Well = data$Well, 
  BurstDuration = data$'Duration..s.', #duration of each burst
  SizeSpikes = data$'Size..spikes.' #number of spikes in each burst
)

## remove junk rows at bottom of excel 
# Identify the row index of the first blank row
blank_row_index <- which(apply(burst_data, 1, function(row) all(row == "")))
# If a blank row exists, subset the data up to that row
if (length(blank_row_index) > 0) {
  burst_data_clean <- burst_data[1:(blank_row_index[1] - 1), ]
}

# Convert necessary columns to numeric data
burst_data_clean$BurstStartTime <- as.numeric(burst_data_clean$BurstStartTime) 
burst_data_clean$BurstDuration <- as.numeric(burst_data_clean$BurstDuration)
burst_data_clean$SizeSpikes <- as.numeric(burst_data_clean$SizeSpikes)


### Vectors for selecting which wells to plot
all_vehicle <- c("A1", "A2", "B1", "B2", "C1", "C2")
all_PTX100nm <- c("A3", "A4", "B3", "B4", "C3", "C4")
all_SPalone <- c("D1", "D2", "E1", "E2", "F1", "F2")
all_PTXwSP <- c("D3", "D4", "E3", "E4", "F3", "F4")

PTX_wells_to_plot <- c("D3", "E2", "B4", "B1") #reverse order, PTX+SP SP PTX veh so that the plot will be veh PTX SP PTX+SP
all_PTX_wells_to_plot <- c("D3", "D4", "E3", "E4", "F3", "F4", "D1", "D2", "E1", "E2", "F1", "F2", "A3", "A4", "B3", "B4", "C3", "C4", "A3", "A4", "B3", "B4", "C3", "C4", "A1", "A2", "B1", "B2", "C1", "C2")
PTX_vs_PTXwSP <- c("D3", "D4", "E3", "E4", "F3", "F4", "A3", "A4", "B3", "B4", "C3", "C4")

### Raster plot for Burst Frequency
ggplot(burst_data_clean, aes(x = BurstStartTime, y = Well)) +
  geom_point(shape = "|", size = 5) +
  scale_y_discrete(limits = PTX_wells_to_plot, # Insert vector here to determine which wells are plotted
                   labels = c("PTX 100nM/SP600125 10µM", "DMSO/SP600125 10µM", "PTX 100nM/DMSO", "DMSO/DMSO")) + 
  scale_x_continuous(limits = c(0, 600), breaks = seq(0, 600, by = 60)) +
  labs(x = "Time(s)", y = "") +
  theme_minimal() +
  ggtitle("Raster Plot of Burst Frequency") + 
  theme(plot.background = element_blank(), 
        panel.grid = element_blank(),  # Remove grid lines
        panel.background = element_blank(), # Remove background
        axis.line.x = element_line(color = "black", size = 0.5))  # Add x-axis line
        # axis.text.x = element_blank(), # Remove x-axis labels
        # axis.ticks.x = element_blank()) # Remove x-axis ticks


### Import data for spikes & clean up data
data <- read.csv("../Documents/R codes/MEA/20240222_264-A_cJUN_Baseline_after_MC(006)(000)_spike_list.csv")
raster_data <- data.frame(
  NeuronID = data$Electrode, 
  SpikeTime = data$'Time..s.'
)

# create a copy of same raster_data dataframe
raster_data_copy <- data.frame(
  NeuronID = data$Electrode, 
  SpikeTime = data$'Time..s.'
)

# create new column of Wells
raster_data_copy$Well <- sub("_.*", "", raster_data$NeuronID)

## remove junk rows at bottom of excel 
# Identify the row index of the first blank row
blank_row_index <- which(apply(raster_data_copy, 1, function(row) all(row == "")))
# If a blank row exists, subset the data up to that row
if (length(blank_row_index) > 0) {
  raster_data_clean <- raster_data_copy[1:(blank_row_index[1] - 1), ]
}

# convert SpikeTime column to numeric
raster_data_clean$SpikeTime <- as.numeric(raster_data_clean$SpikeTime) 


### Raster plot for representative burst
# plotting a specific known burst time frame 
ggplot(raster_data_clean, aes(x = SpikeTime, y = Well)) +
  geom_point(shape = "|", size = 5) +
  scale_y_discrete(limits = PTX_wells_to_plot, 
                   labels = c("PTX 100nM/SP600125 10µM", "DMSO/SP600125 10µM", "DMSO/PTX 100nM", "DMSO/DMSO")
  ) +
  scale_x_continuous(limits = c(284.16752, 286.33256), breaks = seq(284.16752, 286.33256, by = 2),  #input a specific range from burst_data_clean
                     labels = c("284", "310")
  ) +
  labs(x = "Time(s)", y = "") +
  theme_minimal() + # Minimal theme as a base
  ggtitle("Representative Burst") + 
  theme(plot.background = element_blank(), 
        panel.grid = element_blank(),  # Remove grid lines
        panel.background = element_blank(), # Remove background
        # axis.text.x = element_blank(), # Remove x-axis labels
        # axis.ticks.x = element_blank(), # Remove x-axis ticks 
        axis.line.x = element_line(color = "black", size = 0.5))  # Add x-axis line
